/** @file   lightfence.cpp
 * @brief   Implementation of LightFence - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#include "lightfence.h"
#include "poly2d.h"
#include <allegro.h>
using std::vector;
namespace eng2d {

///
/// Static members, constants and datatypes
/// =======================================

const int LightFence::FENCETYPE_LIGHTFENCE;


///
/// Constructors, destructor and operators
/// ======================================
/** Constructor
 */
LightFence::LightFence( const Vec2D& rP1, const Vec2D& rP2 ) :
  Interactive(),
  m_begPos( rP1 ),
  m_endPos( rP2 ),
  m_state( STATE_ON ),
  m_onStateLength( 2000000 ),
  m_offStateLength( 0 ),
  m_stateCounter( 1000000 ),
  m_idcode( 0 ),
  m_pSound( 0 )
{
  // We create the default state sequence.
  StateProp p;
  p.state = STATE_ON;
  p.duration = 1000000;
  m_stateSequence.push_back( p );
  
  m_stateIter = m_stateSequence.begin();
  
  // Define the reference points
  
  Vec2D dirV( m_endPos );
  dirV -= m_begPos;
  dirV.norm();
  dirV *= 10;
  
  Vec2D currentPos( m_begPos );
  while ( true ) {
    m_refpoints.push_back( currentPos );
    
    Vec2D distVec = m_endPos;
    distVec -= currentPos;
    if ( distVec.length() < 10 ) {
      break;
    }
    currentPos += dirV;
  }
}



/** Destructor
 */
LightFence::~LightFence()
{
  for ( int i=0; i < m_wires.size(); i++ ) {
    if ( m_wires.at(i) != 0 ) {
      delete m_wires.at(i);
    }
  }
  m_wires.clear();
}




///
/// Public methods
/// ==============

/** Updates this lightfence
 */
void LightFence::update()
{
  m_stateCounter -= 1;
  if ( m_stateCounter < 0 ) {
    // We should go to next state.
    int tmpState = m_state;
    m_stateIter++;
    if ( m_stateIter == m_stateSequence.end() ) {
      m_stateIter = m_stateSequence.begin();
    }
    m_state = (*m_stateIter).state;
    m_stateCounter = (*m_stateIter).duration;
    if ( m_state != tmpState ) {
      // New state is different from the previous state. If we have a
      // soundsource, we turn it on or off
      if ( m_pSound != 0 && m_state == STATE_ON ) {
        m_pSound->loudness( 128 );
      } else if ( m_pSound != 0 && m_state == STATE_OFF ) {
        m_pSound->loudness( 0 );
      }
    }
  }
    
  if ( m_state == STATE_OFF ) {
    return;
  }
  for ( int i=0; i < m_wires.size(); i++ ) {
    m_wires.at(i)->update();
  }
}



/** Draws the wires that form this fence
 */
void LightFence::redraw( BITMAP* pB, const Vec2D& offset ) const
{
  if ( m_state == STATE_OFF ) {
    return;
  }
  for ( int i=0; i < m_wires.size(); i++ ) {
    m_wires.at(i)->redraw( pB, offset );
  }
}



/** Sets the state sequence
 */
void LightFence::setStateSequence( const vector<StateProp>& stateSequence )
{
  m_stateSequence = stateSequence;
  m_stateIter = m_stateSequence.begin();
  m_state = (*m_stateIter).state;
  m_stateCounter = (*m_stateIter).duration;
}



/** Attaches the given SoundSource to us.
 */
void LightFence::attachSoundSource( SoundSource* pSS )
{
  if ( m_pSound != 0 ) {
    Sound::removeSoundSource( m_pSound );
    m_pSound = 0;
  }
  if ( pSS != 0 ) {
    m_pSound = pSS;
    Sound::createSoundSource( pSS );
  }
}



///
/// Getter methods
/// ==============

/** Tells if given point touches this fence.
 */
bool LightFence::touches( const Vec2D& rP, float maxDistance ) const
{
  if ( m_state == STATE_OFF ) {
    return false;
  }
  
  // If the distance from given point is greater than the distance between
  // one of the end nodes and middle point, then there will not be collision
  
  Vec2D midP( this->middlePos() );
  float halfLen = ( midP - m_begPos ).length2();
  float midDist = (midP - rP).length2();
  if ( midDist > halfLen ) {
    return false;
  }
  
  // Now we need to make closer look in case the point might collide.
  maxDistance *= maxDistance;
  float prevDistance = 10000000;
  float dist = 0;
  for ( int i=0; i < m_refpoints.size(); i++ ) {
    Vec2D tmpVec( m_refpoints.at(i) );
    tmpVec -= rP;
    dist = tmpVec.length2();
    
    if ( dist > prevDistance ) {
      return false;
    }
    if ( dist < maxDistance ) {
      return true;
    }
  }
}



/** Returns the state off this fence
 */
LightFence::State LightFence::state() const
{
  return m_state;
}



/** Returns the soundsource.
 */
SoundSource* LightFence::getSoundSource() const
{
  return m_pSound;
}



/** Returns the begining point
 */
Vec2D LightFence::nodePos1() const
{
  return m_begPos;
}



/** Returns the ending point
 */
Vec2D LightFence::nodePos2() const
{
  return m_endPos;
}



/** Returns the middlepoint
 */
Vec2D LightFence::middlePos() const
{

  Vec2D tmpPos( m_begPos );
  tmpPos += m_endPos;
  tmpPos /= 2;

  return tmpPos;
}



/** Returns the type id.
 */
int LightFence::fenceType() const
{
  return FENCETYPE_LIGHTFENCE;
}



///
/// Private or Protected methods
/// ============================

/** Adds new wire to this fence.
 */
void LightFence::addWire( Wire* pW )
{
  m_wires.push_back( pW );
}


} // end of namespace
